<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use App\Http\Controllers\Traits\OrderStatisticsTrait;
use App\Http\Controllers\Traits\AdminViewSharedDataTrait;
use App\Http\Controllers\Traits\OrderNumberGeneratorTrait;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class ReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    use AdminViewSharedDataTrait;
    use OrderStatisticsTrait;
    use OrderNumberGeneratorTrait;

    public function __construct()
    {
        $this->shareAdminViewData();
        $this->shareOrderStatistics();
    }

    public function index(Request $request, $filter = null)
    {
        if ($filter === 'weekly') {
            $start_date = Carbon::now()->subWeek()->format('Y-m-d');
            $end_date   = Carbon::today()->format('Y-m-d');
        } elseif ($filter === 'monthly') {
            $start_date = Carbon::now()->subMonth()->format('Y-m-d');
            $end_date   = Carbon::today()->format('Y-m-d');
        } elseif ($filter === 'yearly') {
            $start_date = Carbon::now()->subYear()->format('Y-m-d');
            $end_date   = Carbon::today()->format('Y-m-d');
        } else {
            // Default: today or request dates
            $start_date = $request->start_date ?? Carbon::today()->format('Y-m-d');
            $end_date   = $request->end_date ?? Carbon::today()->format('Y-m-d');
        }

        $customerId  = $request->customer_id ?? null;

        // Get all customers for filter dropdown
        $customers = Customer::orderBy('name')->get();

        // Query orders with optional date & customer filter
        $orders = Order::with(['customer', 'orderItems'])
            ->when($start_date && $end_date, function ($q) use ($start_date, $end_date) {
                $q->whereDate('created_at', '>=', $start_date)
                    ->whereDate('created_at', '<=', $end_date);
            })
            ->when($customerId, function ($q) use ($customerId) {
                $q->where('customer_id', $customerId);
            })
            ->get();

        // Prepare report data (per order item)
        $reportData = $orders->flatMap(function ($order) {
            return $order->orderItems->map(function ($item) use ($order) {
                return [
                    'customer_name'   => $order->customer->name ?? '-',
                    'menu_name'       => $item->menu_name,
                    'additional_info' => $order->additional_info ?? '-',
                    'quantity'        => $item->quantity,
                    'total_sales'     => $item->subtotal, // per menu item
                    'created_at'      => $order->created_at,
                ];
            });
        });

        // Totals
        $totalSales    = $reportData->sum('total_sales');
        $totalQuantity = $reportData->sum('quantity');

        // Calculate weekly, monthly, yearly counts for the top cards
        $weeklySalesCount  = Order::where('created_at', '>=', Carbon::now()->subWeek())->count();
        $monthlySalesCount = Order::where('created_at', '>=', Carbon::now()->subMonth())->count();
        $yearlySalesCount  = Order::where('created_at', '>=', Carbon::now()->subYear())->count();

        return view('admin.report', compact(
            'reportData',
            'totalSales',
            'totalQuantity',
            'customers',
            'start_date',
            'end_date',
            'customerId',
            'weeklySalesCount',
            'monthlySalesCount',
            'yearlySalesCount'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
